/*
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1996 David S. Miller (dm@engr.sgi.com)
 * Copyright (C) 1999 Andrew R. Baker (andrewb@uab.edu)
 * Copyright (C) 2001 Florian Lohoff (flo@rfc822.org)
 * Copyright (C) 2003 Ralf Baechle (ralf@linux-mips.org)
 * Copyright (C) 2004 Peter Fuerst (pf@net.alphadv.de) - IP28
 * 
 * (In all truth, Jed Schimmel wrote all this code.)
 */
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/blkdev.h>
#include <linux/version.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <linux/spinlock.h>

#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/sgialib.h>
#include <asm/sgi/sgi.h>
#include <asm/sgi/mc.h>
#include <asm/sgi/hpc3.h>
#include <asm/sgi/ip22.h>
#include <asm/irq.h>
#include <asm/io.h>

#include "scsi.h"
#include <scsi/scsi_host.h>
#include "wd33c93.h"

#include <linux/stat.h>

#if 0
#define DPRINTK(args...)	printk(args)
#else
#define DPRINTK(args...)
#endif

#if defined(CONFIG_SGI_IP28) || defined(CONFIG_SGI_IP26)
/* see SGI documents: 007-0911-210, Chapter I.1. and man ip26_ucmem(D3) */
# define DMA_NONCOHERENT_CODE(code...) code
/* (Extra) code for cached writes to memory with coherency operations. */
# define IP28_IP26_CODE(code...) code
#else
# if defined(CONFIG_DMA_NONCOHERENT) || defined(CONFIG_NONCOHERENT_IO)
#  define DMA_NONCOHERENT_CODE(code...) code
# else
#  define DMA_NONCOHERENT_CODE(code...)
# endif
# define IP28_IP26_CODE(code...)
#endif

#ifdef TO_PHYS /* mach-ip28/spaces.h, mach-ip22/spaces.h/MIPS64, ... */
# define wd93_to_phys(p)  TO_PHYS( (unsigned long)(p))
# define wd93_to_cache(p) TO_CAC(  (unsigned long)(p))
# define wd93_to_uncac(p) TO_UNCAC((unsigned long)(p))
#else /* mach-ip22/spaces.h/MIPS32, ... */
# define wd93_to_phys(p)  CPHYSADDR( p)
# define wd93_to_cache(p) CKSEG0ADDR(p)
# define wd93_to_uncac(p) CKSEG1ADDR(p)
#endif

/* Alignment to the largest IP2X cacheblock. */
#define cachligndn(x)  ((x) & ~0x7f)
#define cachlignup(x)  cachligndn((x)+0x7f)

#define HDATA(ptr) ((struct ip22_hostdata *)((ptr)->hostdata))

struct ip22_hostdata {
	struct WD33C93_hostdata wh;
	struct hpc_data {
		dma_addr_t      dma;
		void            * cpu;
		/*
		 * A cacheline-aligned intermediate dma read buffer for all
		 * the small, not cacheline-aligned Scsi_Cmnd input buffers.
		 */
		char        (*buf)[PAGE_SIZE];
		struct {
			char     *scp;
			char     *rbp;
			unsigned len;
		} rb;
	} hd;
};

struct hpc_chunk {
	struct hpc_dma_desc desc;
	u32 _padding;	/* align to quadword boundary */
};

static dma_addr_t sgiwd93_dma_map(Scsi_Cmnd*);
static void sgiwd93_dma_unmap(Scsi_Cmnd*);

struct Scsi_Host *sgiwd93_host;
struct Scsi_Host *sgiwd93_host1;

/* Wuff wuff, wuff, wd33c93.c, wuff wuff, object oriented, bow wow. */
static inline void write_wd33c93_count(const wd33c93_regs regs,
                                      unsigned long value)
{
	*regs.SASR = WD_TRANSFER_COUNT_MSB;
	mb();
	*regs.SCMD = ((value >> 16) & 0xff);
	*regs.SCMD = ((value >>  8) & 0xff);
	*regs.SCMD = ((value >>  0) & 0xff);
	mb();
}

static inline unsigned long read_wd33c93_count(const wd33c93_regs regs)
{
	unsigned long value;

	*regs.SASR = WD_TRANSFER_COUNT_MSB;
	mb();
	value =  ((*regs.SCMD & 0xff) << 16);
	value |= ((*regs.SCMD & 0xff) <<  8);
	value |= ((*regs.SCMD & 0xff) <<  0);
	mb();
	return value;
}

static irqreturn_t sgiwd93_intr(int irq, void *dev_id, struct pt_regs *regs)
{
	struct Scsi_Host * host = (struct Scsi_Host *) dev_id;
	unsigned long flags;

	spin_lock_irqsave(host->host_lock, flags);
	wd33c93_intr(host);
	spin_unlock_irqrestore(host->host_lock, flags);

	return IRQ_HANDLED;
}

static inline
int fill_hpc_entries(struct hpc_chunk *hcp, Scsi_Cmnd *cmd)
{
	unsigned long len = cmd->SCp.this_residual;
	dma_addr_t physaddr;
	unsigned long count;
	IP28_IP26_CODE( int nc; struct hpc_chunk *hcpstart; )
	

	physaddr = sgiwd93_dma_map(cmd);
	if (!physaddr)
		return 1;

	
	IP28_IP26_CODE(
		hcpstart = hcp = (typeof(hcp)) wd93_to_cache(hcp);
		nc = 1 + (len + 8191) / 8192;
		nc = cachlignup(sizeof(*hcp)*nc);
		/* dma_cache_inv((unsigned long) hcp, nc); */
	)
	while (len) {
		/*
		 * even cntinfo could be up to 16383, without
		 * magic only 8192 works correctly
		 * (even much less than 8192 may fail, since
		 * pbuf must not cross any page boundaries !)
		 */
		count = len > 8192 ? 8192 : len;
		hcp->desc.pbuf = physaddr;
		hcp->desc.cntinfo = count;
		hcp++;
		len -= count;
		physaddr += count;
	}

	/*
	 * To make sure, if we trip an HPC bug, that we transfer every single
	 * byte, we tag on an extra zero length dma descriptor at the end of
	 * the chain.
	 */
	hcp->desc.pbuf = 0;
	hcp->desc.cntinfo = HPCDMA_EOX;

	IP28_IP26_CODE( dma_cache_wback_inv((unsigned long)hcpstart, nc); )
	
	return 0;
}

static int dma_setup(Scsi_Cmnd *cmd, int datainp)
{
	struct ip22_hostdata *hdata = HDATA(cmd->device->host);
	struct hpc3_scsiregs *hregs =
		(struct hpc3_scsiregs *) cmd->device->host->base;
	struct hpc_chunk *hcp = (struct hpc_chunk *) hdata->hd.cpu;

	DPRINTK("dma_setup: datainp<%d> hcp<%p> ", datainp, hcp);

	if (datainp && cmd->sc_data_direction != DMA_FROM_DEVICE ||
		!datainp && cmd->sc_data_direction != DMA_TO_DEVICE)
		DPRINTK(KERN_WARNING "sgiwd93: DMA direction mismatch: "
			"data %sput vs. sc_data_direction %d !\n",
			datainp ? "in":"out", cmd->sc_data_direction);
	hdata->wh.dma_dir = datainp;

	/*
	 * wd33c93 shouldn't pass us bogus dma_setups, but it does:-(  The
	 * other wd33c93 drivers deal with it the same way (which isn't that
	 * obvious).  IMHO a better fix would be, not to do these dma setups
	 * in the first place.
	 */
	if (cmd->SCp.ptr == NULL || cmd->SCp.this_residual == 0) {
		DPRINTK("\n");
		return 1;
	}
	if (fill_hpc_entries(hcp, cmd)) {
		DPRINTK("\n");
		return 1;
	}
	DPRINTK(" HPCGO\n");

	/* Start up the HPC. */
	hregs->ndptr = hdata->hd.dma;
	if (datainp)
		hregs->ctrl = HPC3_SCTRL_ACTIVE;
	else
		hregs->ctrl = HPC3_SCTRL_ACTIVE | HPC3_SCTRL_DIR;

	return 0;
}

static void dma_stop(struct Scsi_Host *instance, Scsi_Cmnd *SCpnt,
		     int status)
{
	struct ip22_hostdata *hdata = HDATA(instance);
	struct hpc3_scsiregs *hregs;

	if (!SCpnt)
		return; /* really ignore wd33c93_host_reset() ? */

	hregs = (struct hpc3_scsiregs *) SCpnt->device->host->base;

	DPRINTK("dma_stop: status<%d> ", status);

	/* First stop the HPC and flush it's FIFO. */
	if (hdata->wh.dma_dir) {
		hregs->ctrl |= HPC3_SCTRL_FLUSH;
		while (hregs->ctrl & HPC3_SCTRL_ACTIVE)
			barrier();
	}
	hregs->ctrl = 0;
	sgiwd93_dma_unmap(SCpnt);

	DPRINTK("\n");
}

void sgiwd93_reset(unsigned long base)
{
	struct hpc3_scsiregs *hregs = (struct hpc3_scsiregs *) base;

	hregs->ctrl = HPC3_SCTRL_CRESET;
	udelay(50);
	hregs->ctrl = 0;
}

static inline void init_hpc_chain(struct hpc_data *hd)
{
	struct hpc_chunk *hcp = (struct hpc_chunk *) hd->cpu;
	struct hpc_chunk *dma = (struct hpc_chunk *) hd->dma;
	unsigned long start, end;
	

	IP28_IP26_CODE( hcp = (typeof(hcp)) wd93_to_cache(hcp); )
	start = (unsigned long) hcp;
	end = start + PAGE_SIZE;

	while (start < end) {
		hcp->desc.pnext = (u32) (dma + 1);
		hcp->desc.cntinfo = HPCDMA_EOX;
		hcp++; dma++;
		start += sizeof(struct hpc_chunk);
	}
	hcp--;
	hcp->desc.pnext = hd->dma;

	IP28_IP26_CODE(
		/* Force flush to memory and purge cached references ! */
		dma_cache_wback_inv(wd93_to_cache(hd->cpu), PAGE_SIZE);
	)
	
}

static struct Scsi_Host * __init sgiwd93_setup_scsi(
	Scsi_Host_Template *SGIblows, int unit, int irq,
	struct hpc3_scsiregs *hregs, unsigned char *wdregs)
{
	struct ip22_hostdata *hdata;
	struct Scsi_Host *host;
	wd33c93_regs regs;

	host = scsi_register(SGIblows, sizeof(struct ip22_hostdata));
	if (!host)
		return NULL;

	host->base = (unsigned long) hregs;
	host->irq = irq;

	hdata = HDATA(host);
	hdata->hd.cpu = dma_alloc_coherent(NULL, DMA_NONCOHERENT_CODE(2*)PAGE_SIZE,
	                                   &hdata->hd.dma, GFP_KERNEL);
	if (!hdata->hd.cpu) {
		printk(KERN_WARNING "sgiwd93: Could not allocate memory for "
		       "host %d buffer.\n", unit);
		goto out_unregister;
	}
	hdata->hd.rb.rbp = 0;
	hdata->hd.rb.scp = 0;
	hdata->hd.rb.len = 0;
	hdata->hd.buf = 0;
	DMA_NONCOHERENT_CODE(
		hdata->hd.buf = (void*) ((char*)hdata->hd.cpu + PAGE_SIZE);
	)
	init_hpc_chain(&hdata->hd);

	regs.SASR = wdregs + 3;
	regs.SCMD = wdregs + 7;

	wd33c93_init(host, regs, dma_setup, dma_stop, WD33C93_FS_MHZ(20));

	if (0xff == hdata->wh.no_sync)
		hdata->wh.no_sync = 0;

	if (request_irq(irq, sgiwd93_intr, 0, "SGI WD93", (void *) host)) {
		printk(KERN_WARNING "sgiwd93: Could not register irq %d "
		       "for host %d.\n", irq, unit);
		goto out_free;
	}
	return host;

out_free:
	dma_free_coherent(NULL, DMA_NONCOHERENT_CODE(2*)PAGE_SIZE,
	                  hdata->hd.cpu, hdata->hd.dma);
	wd33c93_release();

out_unregister:
	scsi_unregister(host);

	return NULL;
}

int __init sgiwd93_detect(Scsi_Host_Template *SGIblows)
{
	int found = 0;

	SGIblows->proc_name = "SGIWD93";
	sgiwd93_host = sgiwd93_setup_scsi(SGIblows, 0, SGI_WD93_0_IRQ,
	                                  &hpc3c0->scsi_chan0,
	                                  (unsigned char *)hpc3c0->scsi0_ext);
	if (sgiwd93_host)
		found++;

	/* Set up second controller on the Indigo2 */
	if (ip22_is_fullhouse()) {
		sgiwd93_host1 = sgiwd93_setup_scsi(SGIblows, 1, SGI_WD93_1_IRQ,
		                          &hpc3c0->scsi_chan1,
		                          (unsigned char *)hpc3c0->scsi1_ext);
		if (sgiwd93_host1)
			found++;
	}

	return found;
}

int sgiwd93_release(struct Scsi_Host *instance)
{
	struct ip22_hostdata *hdata = HDATA(instance);
	int irq = 0;

	if (sgiwd93_host && sgiwd93_host == instance)
		irq = SGI_WD93_0_IRQ;
	else if (sgiwd93_host1 && sgiwd93_host1 == instance)
		irq = SGI_WD93_1_IRQ;

	free_irq(irq, sgiwd93_intr);
	dma_free_coherent(NULL, DMA_NONCOHERENT_CODE(2*)PAGE_SIZE,
	                  hdata->hd.cpu, hdata->hd.dma);
	wd33c93_release();

	return 1;
}

static int sgiwd93_bus_reset(Scsi_Cmnd *cmd)
{
	/* FIXME perform bus-specific reset */

	/* FIXME 2: kill this function, and let midlayer fallback
	   to the same result, calling wd33c93_host_reset() */

	spin_lock_irq(cmd->device->host->host_lock);
	wd33c93_host_reset(cmd);
	spin_unlock_irq(cmd->device->host->host_lock);

	return SUCCESS;
}

static dma_addr_t sgiwd93_dma_map( Scsi_Cmnd *cmd )
{
	struct ip22_hostdata *hdata = HDATA(cmd->device->host);
	unsigned len = cmd->SCp.this_residual;
	void *addr = cmd->SCp.ptr;
	/*
	 * We cannot use 'cmd->sc_data_direction' here, since we are called
	 * for an explicit dma-direction, and it is mere *concidence*, if
	 * 'sc_data_direction' is consistent with this direction !
	 * (tracing shows, 'sc_...' isn't well defined at all...)
	 */
	int dmadir = hdata->wh.dma_dir ? DMA_FROM_DEVICE:DMA_TO_DEVICE;

	DMA_NONCOHERENT_CODE(
	if (hdata->hd.rb.rbp) {
		printk(KERN_ERR "sgiwd93_dma_map: %s: discarding stale input buffer!"
			" (%p)\n", hdata->wh.dma_dir ? "in":"out", hdata->hd.rb.scp);
		if (hdata->hd.rb.rbp != hdata->hd.buf)
			kfree(hdata->hd.rb.rbp);
	}
	hdata->hd.rb.scp = addr;
	hdata->hd.rb.rbp = 0;
	hdata->hd.rb.len = len;
	if (hdata->wh.dma_dir) {
		/*
		 * (at least by now) it seems, we cannot safely avoid
		 * to intercept *all* input-buffers on IP28.
		 */
		if (IP28_IP26_CODE(1 || )cachlignup(len) != len ||
		    cachligndn((unsigned)addr) != (unsigned)addr) {
			len = cachlignup(len);
			if (len <= sizeof(*hdata->hd.buf)) {
				hdata->hd.rb.rbp = addr = hdata->hd.buf;
				/* Sigh... Screw it, to make dma_map_single() work. */
				addr = phys_to_virt( wd93_to_phys(addr) );
			} else {
				if ( !(addr = kmalloc(len, GFP_KERNEL)) )
					return (cmd->SCp.dma_handle = 0);
				hdata->hd.rb.rbp = addr;
			}
		}
	})
	cmd->SCp.dma_handle = dma_map_single(NULL, addr, len, dmadir);
	return cmd->SCp.dma_handle;
}

static void sgiwd93_dma_unmap( Scsi_Cmnd *cmd )
{
	struct ip22_hostdata *hdata = HDATA(cmd->device->host);
	unsigned len = cmd->SCp.this_residual;

#ifdef CONFIG_CPU_R10000
	/* R10000 also requires cache invalidation after DMA ended */
	DMA_NONCOHERENT_CODE(
		if (hdata->hd.rb.rbp)
			dma_cache_inv(wd93_to_cache(hdata->hd.rb.rbp),
			              cachlignup(hdata->hd.rb.len));
		else if (hdata->wh.dma_dir)
			dma_cache_inv((unsigned long)cmd->SCp.ptr, len);
	)
#endif
	DMA_NONCOHERENT_CODE(
		/* We had dma_map'ed an aligned input buffer ! */
		if (hdata->wh.dma_dir) len = cachlignup(len);
	)
	dma_unmap_single(NULL, cmd->SCp.dma_handle, len,
	                 hdata->wh.dma_dir ? DMA_FROM_DEVICE:DMA_TO_DEVICE);

	DMA_NONCOHERENT_CODE(
	if (cmd->SCp.ptr != hdata->hd.rb.scp)
		printk(KERN_ERR "sgiwd93_dma_map: %s: mismatched SCp! (%p,%p)\n",
			hdata->wh.dma_dir ? "in":"out", hdata->hd.rb.scp,
			cmd->SCp.ptr);
	if (hdata->hd.rb.rbp) {
		if (!hdata->wh.dma_dir)
			printk(KERN_ERR "sgiwd93_dma_unmap: out: discarding "
				"stale input buffer! (%p)\n", hdata->hd.rb.scp);
		else /* if (cmd->SCp.ptr == hdata->hd.rb.scp) */
			memcpy(hdata->hd.rb.scp, wd93_to_cache(hdata->hd.rb.rbp),
			       hdata->hd.rb.len);

		if (hdata->hd.rb.rbp != hdata->hd.buf)
			kfree(hdata->hd.rb.rbp);
	}
	hdata->hd.rb.scp = 0;
	hdata->hd.rb.rbp = 0;
	hdata->hd.rb.len = 0;
	)
}

/*
 * Kludge alert - the SCSI code calls the abort and reset method with int
 * arguments not with pointers.  So this is going to blow up beautyfully
 * on 64-bit systems with memory outside the compat address spaces.
 */
static Scsi_Host_Template driver_template = {
	.name			= "SGI WD93",
	.detect			= sgiwd93_detect,
	.release		= sgiwd93_release,
	.queuecommand		= wd33c93_queuecommand,
	.eh_abort_handler	= wd33c93_abort,
	.eh_bus_reset_handler	= sgiwd93_bus_reset,
	.eh_host_reset_handler	= wd33c93_host_reset,
	.proc_info		= wd33c93_proc_info,
	.proc_name		= "SGIWD93",
	.can_queue		= 16,
	.this_id		= 7,
	.sg_tablesize		= SG_ALL,
	.cmd_per_lun		= 8,
	.use_clustering		= DISABLE_CLUSTERING,
};
#include "scsi_module.c"
