/*
 * welcomescreen.c	Wed Nov  2 20:21:59 2011
 * (impact_driver.c	2005/07/12 23:24:15, Copyright (c) 2005 peter fuerst)
 *
 * Retreive the (welcome-)screenimage, captured by the Impact frame buffer
 * device driver during startup, and store it away in files for further use.
 */
#include <stdio.h>
#include <stdint.h>	/* uint32_t... */
#include <stdlib.h>	/* mallloc */
#include <errno.h>	/* ENOMEM */
#include <sys/types.h>	/* open */
#include <sys/stat.h>	/* open */
#include <fcntl.h>	/* open */
#include <unistd.h>	/* m[un]map */
#include <sys/mman.h>	/* m[un]map */
#include <string.h>	/* strrchr */

#define IMPACT_FB_MMAP_OFF(pool)	((pool)<<24)
#define POOL	2	/* Don't want to interfere with a running Xserver */

static char *ImpactMapShadowFB(size_t size, int pool)
{
	int devFD = open("/dev/fb0",O_RDWR);

	if (devFD < 0)
		perror("failed to open /dev/fb0");
	else
	{	char *ShadowPtr =
			mmap((void*)0, size, PROT_READ|PROT_WRITE,
					MAP_SHARED, devFD, IMPACT_FB_MMAP_OFF(pool));
		close(devFD);
		if (ShadowPtr != MAP_FAILED)
			fprintf(stderr, "mapped %ukB @ %p\n", size/1024, ShadowPtr);
		else
		{	char buf[80];
			sprintf(buf, "could not mmap shadow buffer (%p,0x%xkB)",
					(void*)IMPACT_FB_MMAP_OFF(pool), size/1024);
			perror(buf);
		}
		return ShadowPtr;
	}
	return 0;
}

static void ImpactUnmapShadowFB(char *ShadowPtr, size_t size, int pool)
{
	if (ShadowPtr)
	{	char *dummy;
		munmap(ShadowPtr, size);
		/* Then deallocate (most of) the memory. */
		if ((dummy = ImpactMapShadowFB(1,pool)))
			munmap(dummy, 1);
	}
}

/* An interval (inside a pixel-line) of pixels of the same color. */
struct pixcol_range { uint32_t c; uint16_t first; uint16_t last; };

static unsigned reducedindex(unsigned rgba)
{
	/* significant bits 23..22, 15..13, 7..5 */
	return (rgba>>16) & 0xc0 | (rgba>>10) & 0x38 | (rgba>>5) & 7;
}

static int writefile(const char *buf, unsigned sz, const char *file)
{
	FILE *fp = fopen(file,"w"); 
	int r = -1;

	if (fp)
	{	r = fwrite(buf,sz,1,fp) != 1 ? -1:0;
		fclose(fp);
	}
	if (r) perror(file);
	return r;
}

static char *strxdup(const char *s, int n)
{
	char *p = malloc(strlen(s) + n + 1);
	if (p) strcpy(p, s);
	return p;
}

static void output(char *buf, const char *file, int translate)
{
	const uint32_t *idx = (uint32_t*)buf;
	const uint32_t *col = idx + 1025 + 2;
	struct pixcol_range *dat = (struct pixcol_range*)(col + 256 + 2);
	char *p;

	if (memcmp(buf,"Impact",6))
	{	fprintf(stderr, "'Impact' signature not found\n");
		return;
	}
	if (memcmp(col,"Colortable",8))
		if (!memcmp(col,"Pixeldata",8)) /* Already translated. */
			dat = (struct pixcol_range*)col;
		else
		{	fprintf(stderr, "'Colortable' signature not found\n");
			return;
		}
	if (memcmp(dat,"Pixeldata",8))
	{	fprintf(stderr, "'Pixeldata' signature not found\n");
		return;
	}
	if (!(p = strxdup(file,10)))
	{	fprintf(stderr, "%s: %s\n", file, strerror(ENOMEM));
		return;
	}
	file = p;
	p = strchr(p,0);
	idx += 2;
	col += 2;
	dat++;

	strcpy(p,".colors");
	if ((void*)col != dat)
		if (translate)
		{	int i;
			for (i = 0; i < idx[1024]; i++)
				dat[i].c = col[reducedindex(dat[i].c)];
		}
		else if (!writefile((void*)col,sizeof(*col)*256,file))
			fprintf(stderr, "%s:\t256 entries, 32bit, MSB 1st\n", file);

	strcpy(p,".index");
	if (!writefile((void*)idx,sizeof(*idx)*1025,file))
		fprintf(stderr, "%s:\t1024+1 entries, 32bit, MSB 1st\n", file);
	
	strcpy(p,".data");
	if (!writefile((void*)dat,sizeof(*dat)*idx[1024],file))
		fprintf(stderr, "%s:\t%d entries, {32,16,16bit} ABGR, MSB 1st...\n",
			file, idx[1024]);
	
	free((void*)file);
}

static void usage ( const char *argv0 )
{
	const char *p = strrchr(argv0,'/');
	p = p ? p+1:argv0;
	fprintf(stderr, "usage: %s [-c ][-s size] [-o ]outfile\n"
		"\t" "-c" "\tapply colortable on output\n"
		"\t" "-s N" "\tallocate N kbytes (default is 512)\n\n"
		"output will be distributed to three (two with '-c') files:\n"
		"outfile.index outfile.data [outfile.colors]\n", p);
	exit(1);
}

int main ( int argc, char *argv[] )
{
	char *ShadowPtr;
	const char *outfile = 0;
	size_t sz = 512*1024;
	int i, translate = 0;

	for (i = 1; i < argc; i++)
	{	if ('-' == *argv[i])
			switch (argv[i][1])
			{	case 'c':
					if (argv[i][2])
						usage(argv[0]);
					translate++;
					break;
				case 's':
					if (argv[i][2])
						sz = 1024 * strtol(argv[i]+2,0,0);
					else if (i+1 < argc)
						sz = 1024 * strtol(argv[++i],0,0);
					else
						usage(argv[0]);
					break;
				case 'o':
					if (outfile)
						usage(argv[0]);
					if (argv[i][2])
						outfile = argv[i]+2;
					else if (i+1 < argc)
						outfile = argv[++i];
					else
						usage(argv[0]);
					break;
				default:
					usage(argv[0]);
			}
		else if (outfile)
			usage(argv[0]);
		else
			outfile = argv[i];
	}
	if (!outfile) usage(argv[0]);

	if (sz < 8192) sz = 8192;	/* sz > sizeof(int)*1025 */
	
	if ((ShadowPtr = ImpactMapShadowFB(sz,POOL)))
	{	output(ShadowPtr, outfile, translate);
		ImpactUnmapShadowFB(ShadowPtr,sz,POOL);
	}
}

/* eof */
